using System;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class is a static module containing shared implementation units
	/// for VistALink security messages. It is used to model/fake multiple implementation inheritance. 
	/// </summary>
	internal sealed class SecurityMessageImplementationMixin
	{	
		private static readonly VersionNumber SecurityVersion = new VersionNumber( "1.0" );

		// Constants used in XML message serialization/deserialization		
		private const string XMLCONSTS_SECURITY_INFO_NODE_NAME = "SecurityInfo";
		private const string XMLCONSTS_VERSION_SI_ATTRIBUTE_NAME = "version";

		private SecurityMessageImplementationMixin() {}

		/// <summary>
		/// Empty static constructor added to get rid of "beforefieldinit" attribute generated by compiler.
		/// </summary>
		static SecurityMessageImplementationMixin() {}

		/// <summary>
		/// VistALink security messages shared XML serialization code. It should be called 
		/// from corresponding XML serialization methods of VistALink security message classes. 
		/// </summary>
		/// <param name="writer">XmlWriter to use</param>
		public static void WriteCommonSecurityXmlContent( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_SECURITY_INFO_NODE_NAME );
				writer.WriteAttributeString( XMLCONSTS_VERSION_SI_ATTRIBUTE_NAME, SecurityVersion.ToString() );
			writer.WriteEndElement();
		}

		/// <summary>
		/// VistALink security messages shared XML deserialization code. It should be called 
		/// from XML deserialization constructors of VistALink security message classes. 
		/// </summary>
		/// <param name="sourceDoc">Source Xml document</param>
		public static void ParseCommonSecurityXmlContent( XmlDocument sourceDoc )
		{
			if( sourceDoc == null )
				throw( new ArgumentNullException( "sourceDoc" ) );

			XmlElement _securityInfoElement = (XmlElement)sourceDoc.DocumentElement.FirstChild; // SecurityInfo must be immediately before the request element

			if( _securityInfoElement == null || _securityInfoElement.Name != XMLCONSTS_SECURITY_INFO_NODE_NAME ) 
				throw( new XmlParseException( SR.Exceptions.XmlParseRequiredElementNotFound( XMLCONSTS_SECURITY_INFO_NODE_NAME )) );
			
			// Security version is optional
			if( !_securityInfoElement.HasAttribute( XMLCONSTS_VERSION_SI_ATTRIBUTE_NAME ) )
				return;

			// Message security version must be higher or equal to the supported version
			if( SecurityVersion.CompareTo( VersionNumber.Parse( _securityInfoElement.GetAttribute( XMLCONSTS_VERSION_SI_ATTRIBUTE_NAME ) ) ) > 0 )
				throw( new XmlParseException( SR.Exceptions.VistALinkSecurityVersionMismatch( 
					_securityInfoElement.GetAttribute( XMLCONSTS_VERSION_SI_ATTRIBUTE_NAME ), 
					SecurityVersion.ToString() ) ) );
		}
	}
}
